<?php
require_once 'captcha_system.php';

header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'error' => 'Method not allowed']);
    exit;
}

if (!isset($_POST['action']) || $_POST['action'] !== 'verify_hcaptcha') {
    echo json_encode(['success' => false, 'error' => 'Invalid action']);
    exit;
}

// Anti-bot checks
$user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';
$suspicious_agents = ['bot', 'crawler', 'spider', 'scraper', 'automated'];

foreach ($suspicious_agents as $agent) {
    if (stripos($user_agent, $agent) !== false) {
        error_log("Blocked suspicious user agent: " . $user_agent);
        echo json_encode(['success' => false, 'error' => 'Access denied']);
        exit;
    }
}

// Rate limiting
session_start();
$rate_limit_key = 'hcaptcha_attempts_' . hash('sha256', $_SERVER['REMOTE_ADDR']);

if (!isset($_SESSION[$rate_limit_key])) {
    $_SESSION[$rate_limit_key] = ['count' => 0, 'last_attempt' => time()];
}

$rate_data = $_SESSION[$rate_limit_key];
$time_window = 300; // 5 minutes
$max_attempts = 10;

if (time() - $rate_data['last_attempt'] < $time_window) {
    if ($rate_data['count'] >= $max_attempts) {
        error_log("Rate limit exceeded for IP: " . $_SERVER['REMOTE_ADDR']);
        echo json_encode(['success' => false, 'error' => 'Too many attempts. Please wait before trying again.']);
        exit;
    }
    $_SESSION[$rate_limit_key]['count']++;
} else {
    $_SESSION[$rate_limit_key] = ['count' => 1, 'last_attempt' => time()];
}

try {
    $hcaptcha = new HCaptchaSystem();
    
    $token = $_POST['h-captcha-response'] ?? '';
    
    if (empty($token)) {
        error_log("Missing hCaptcha token");
        echo json_encode(['success' => false, 'error' => 'Please complete the captcha']);
        exit;
    }
    
    $result = $hcaptcha->verifyHCaptcha($token);
    
    if ($result['success']) {
        // Reset rate limiting on successful verification
        unset($_SESSION[$rate_limit_key]);
        error_log("hCaptcha verification successful for IP: " . $_SERVER['REMOTE_ADDR']);
    } else {
        error_log("hCaptcha verification failed: " . $result['error']);
    }
    
    echo json_encode($result);
    
} catch (Exception $e) {
    error_log("hCaptcha verification error: " . $e->getMessage());
    echo json_encode(['success' => false, 'error' => 'Verification error: ' . $e->getMessage()]);
}
?>