// Simple Captcha JavaScript with Smart Anti-Inspection
class SimpleCaptchaHandler {
    constructor() {
        this.startTime = Date.now();
        this.devToolsWarningShown = false;
        this.init();
        // Only setup basic protection during captcha phase
        this.setupBasicProtection();
    }
    
    init() {
        document.addEventListener('DOMContentLoaded', () => {
            this.setupEventListeners();
            this.addHoneypots();
        });
    }
    
    setupBasicProtection() {
        // Only disable right-click during captcha phase
        document.addEventListener('contextmenu', (e) => {
            e.preventDefault();
            this.showMildWarning();
            return false;
        });
        
        // Only block obvious inspection shortcuts during captcha
        document.addEventListener('keydown', (e) => {
            // Only block F12 and Ctrl+U (view source) during captcha
            if (e.keyCode === 123 || (e.ctrlKey && e.keyCode === 85)) {
                e.preventDefault();
                this.showMildWarning();
                return false;
            }
        });
    }
    
    setupFullAntiInspection() {
        // This runs ONLY after captcha is solved
        console.log('Activating full anti-inspection protection...');
        
        // Now block all developer tool shortcuts
        document.addEventListener('keydown', (e) => {
            if (e.keyCode === 123 || // F12
                (e.ctrlKey && e.shiftKey && e.keyCode === 73) || // Ctrl+Shift+I
                (e.ctrlKey && e.keyCode === 85) || // Ctrl+U
                (e.ctrlKey && e.shiftKey && e.keyCode === 67) || // Ctrl+Shift+C
                (e.ctrlKey && e.shiftKey && e.keyCode === 74)) { // Ctrl+Shift+J
                e.preventDefault();
                this.handleDevToolsAttempt();
                return false;
            }
        });
        
        // Detect DevTools opening
        let devtools = {open: false, orientation: null};
        const threshold = 160;
        
        const checkDevTools = () => {
            if (window.outerHeight - window.innerHeight > threshold || 
                window.outerWidth - window.innerWidth > threshold) {
                if (!devtools.open) {
                    devtools.open = true;
                    this.handleDevToolsOpen();
                }
            } else {
                devtools.open = false;
            }
        };
        
        // Start monitoring after a short delay
        setTimeout(() => {
            setInterval(checkDevTools, 1000);
        }, 2000);
        
        // Disable text selection
        document.onselectstart = () => false;
        document.onmousedown = () => false;
        document.ondragstart = () => false;
        
        // Add CSS protection
        const style = document.createElement('style');
        style.textContent = `
            body, * {
                -webkit-user-select: none !important;
                -moz-user-select: none !important;
                -ms-user-select: none !important;
                user-select: none !important;
                -webkit-touch-callout: none !important;
                -webkit-tap-highlight-color: transparent !important;
            }
        `;
        document.head.appendChild(style);
        
        // Console protection
        let consoleWarning = false;
        Object.defineProperty(window, 'console', {
            get: function() {
                if (!consoleWarning) {
                    consoleWarning = true;
                    setTimeout(() => {
                        document.body.innerHTML = '<div style="text-align:center;padding:50px;font-family:Arial;"><h1>🚫 Access Denied</h1><p>Console access detected. Code inspection is not allowed.</p><button onclick="location.reload()" style="padding:10px 20px;background:#4285f4;color:white;border:none;border-radius:5px;cursor:pointer;margin-top:20px;">Reload Page</button></div>';
                    }, 1000);
                }
                return console;
            }
        });
    }
    
    handleDevToolsOpen() {
        document.body.innerHTML = `
            <div style="text-align:center;padding:50px;font-family:Arial;background:#f5f5f5;min-height:100vh;">
                <h1 style="color:#d32f2f;">🚫 Access Denied</h1>
                <p style="font-size:18px;color:#666;">Developer tools detected.</p>
                <p style="color:#666;">Code inspection is not allowed on this page.</p>
                <button onclick="location.reload()" style="padding:10px 20px;background:#4285f4;color:white;border:none;border-radius:5px;cursor:pointer;margin-top:20px;">Reload Page</button>
            </div>
        `;
    }
    
    handleDevToolsAttempt() {
        if (!this.devToolsWarningShown) {
            this.devToolsWarningShown = true;
            this.showStrongWarning();
            setTimeout(() => {
                this.handleDevToolsOpen();
            }, 3000);
        } else {
            this.handleDevToolsOpen();
        }
    }
    
    showMildWarning() {
        const warning = document.createElement('div');
        warning.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            background: #ff9800;
            color: white;
            padding: 10px 15px;
            border-radius: 5px;
            z-index: 10000;
            font-family: Arial;
            font-size: 14px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.3);
        `;
        warning.textContent = '⚠️ Please complete verification first';
        document.body.appendChild(warning);
        
        setTimeout(() => {
            if (warning.parentNode) {
                warning.parentNode.removeChild(warning);
            }
        }, 2000);
    }
    
    showStrongWarning() {
        const warning = document.createElement('div');
        warning.style.cssText = `
            position: fixed;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            background: #d32f2f;
            color: white;
            padding: 20px 30px;
            border-radius: 10px;
            z-index: 10000;
            font-family: Arial;
            text-align: center;
            box-shadow: 0 8px 24px rgba(0,0,0,0.5);
        `;
        warning.innerHTML = `
            <h3 style="margin: 0 0 10px 0;">🚫 Warning</h3>
            <p style="margin: 0;">Developer tools access detected!<br>Redirecting in 3 seconds...</p>
        `;
        document.body.appendChild(warning);
        
        setTimeout(() => {
            if (warning.parentNode) {
                warning.parentNode.removeChild(warning);
            }
        }, 3000);
    }
    
    setupEventListeners() {
        const answerInput = document.getElementById('captcha-answer');
        const checkbox = document.getElementById('captcha-checkbox');
        const verifyBtn = document.getElementById('verify-captcha');
        
        // Get the correct answer from the inline script
        const correctAnswer = window.correctAnswer || '';
        
        if (answerInput && checkbox && verifyBtn) {
            const checkInputs = () => {
                const userAnswer = answerInput.value.trim();
                const isCorrectAnswer = userAnswer === correctAnswer;
                const isChecked = checkbox.checked;
                const shouldEnable = isCorrectAnswer && isChecked;
                
                verifyBtn.disabled = !shouldEnable;
                
                // Force visual feedback
                if (shouldEnable) {
                    verifyBtn.style.background = '#4285f4';
                    verifyBtn.style.cursor = 'pointer';
                    verifyBtn.style.opacity = '1';
                    verifyBtn.classList.remove('disabled');
                    verifyBtn.classList.add('enabled');
                } else {
                    verifyBtn.style.background = '#ccc';
                    verifyBtn.style.cursor = 'not-allowed';
                    verifyBtn.style.opacity = '0.6';
                    verifyBtn.classList.remove('enabled');
                    verifyBtn.classList.add('disabled');
                }
                
                // Visual feedback for answer correctness
                if (userAnswer && !isCorrectAnswer) {
                    answerInput.style.borderColor = '#f44336';
                    answerInput.style.backgroundColor = '#ffebee';
                } else if (isCorrectAnswer) {
                    answerInput.style.borderColor = '#4CAF50';
                    answerInput.style.backgroundColor = '#e8f5e8';
                } else {
                    answerInput.style.borderColor = '#ddd';
                    answerInput.style.backgroundColor = 'white';
                }
            };
            
            // Initial check
            setTimeout(checkInputs, 100);
            
            // Add multiple event listeners for reliability
            answerInput.addEventListener('input', checkInputs);
            answerInput.addEventListener('keyup', checkInputs);
            answerInput.addEventListener('change', checkInputs);
            answerInput.addEventListener('blur', checkInputs);
            
            checkbox.addEventListener('change', checkInputs);
            checkbox.addEventListener('click', checkInputs);
            
            // Force check every second as backup
            setInterval(checkInputs, 1000);
            
            // Add click handler for verify button
            verifyBtn.addEventListener('click', (e) => {
                e.preventDefault();
                e.stopPropagation();
                
                if (!verifyBtn.disabled) {
                    this.verifyCaptcha();
                } else {
                    // Force re-check
                    checkInputs();
                }
            });
            
            // Allow Enter key to submit if button is enabled
            answerInput.addEventListener('keypress', (e) => {
                if (e.key === 'Enter') {
                    e.preventDefault();
                    checkInputs(); // Force check first
                    if (!verifyBtn.disabled) {
                        this.verifyCaptcha();
                    }
                }
            });
            
            // Also check when checkbox label is clicked
            const checkboxLabel = document.querySelector('.checkbox-container');
            if (checkboxLabel) {
                checkboxLabel.addEventListener('click', () => {
                    setTimeout(checkInputs, 50); // Small delay for checkbox state change
                });
            }
        }
    }
    
    addHoneypots() {
        const honeypots = [
            '<input type="text" name="website" class="honeypot" tabindex="-1" autocomplete="off">',
            '<input type="email" name="email_confirm" class="honeypot" tabindex="-1" autocomplete="off">'
        ];
        
        const container = document.querySelector('.captcha-container');
        if (container) {
            honeypots.forEach(honeypot => {
                container.insertAdjacentHTML('beforeend', honeypot);
            });
        }
    }
    
    async verifyCaptcha() {
        const answerInput = document.getElementById('captcha-answer');
        const checkbox = document.getElementById('captcha-checkbox');
        const token = document.querySelector('[name="captcha_token"]');
        
        if (!answerInput || !checkbox || !token) {
            this.showError('Missing form elements');
            return;
        }
        
        const answer = answerInput.value.trim();
        const isChecked = checkbox.checked;
        
        if (!answer) {
            this.showError('Please enter an answer');
            return;
        }
        
        if (!isChecked) {
            this.showError('Please check the "I\'m not a robot" box');
            return;
        }
        
        // Check honeypots
        const honeypots = document.querySelectorAll('.honeypot');
        for (let honeypot of honeypots) {
            if (honeypot.value.trim() !== '') {
                this.showError('Bot detected');
                return;
            }
        }
        
        this.showLoading();
        
        try {
            const formData = new FormData();
            formData.append('action', 'verify_captcha');
            formData.append('answer', answer);
            formData.append('checkbox', isChecked ? '1' : '0');
            formData.append('token', token.value);
            
            const response = await fetch('captcha_verify.php', {
                method: 'POST',
                body: formData
            });
            
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            
            const result = await response.json();
            
            if (result.success) {
                this.showSuccess();
                // Activate full protection AFTER successful captcha
                setTimeout(() => {
                    this.setupFullAntiInspection();
                    window.location.reload();
                }, 1500);
            } else {
                this.showError(result.error || 'Verification failed');
            }
        } catch (error) {
            this.showError('Network error. Please try again.');
        }
    }
    
    showLoading() {
        const container = document.querySelector('.captcha-container');
        container.innerHTML = `
            <div class="captcha-loading" style="display: block;">
                <div class="loading-spinner"></div>
                <p>Verifying...</p>
            </div>
        `;
    }
    
    showSuccess() {
        const container = document.querySelector('.captcha-container');
        container.innerHTML = `
            <div class="captcha-success">
                <div style="font-size: 48px; margin-bottom: 20px;">✓</div>
                <h3>Verification Successful!</h3>
                <p>Loading secure content...</p>
            </div>
        `;
    }
    
    showError(message) {
        const container = document.querySelector('.captcha-container');
        container.innerHTML = `
            <div class="captcha-error">
                <div style="font-size: 48px; margin-bottom: 20px;">✗</div>
                <h3>Verification Failed</h3>
                <p>${message}</p>
                <button onclick="location.reload()" style="margin-top: 20px; padding: 10px 20px; background: #4285f4; color: white; border: none; border-radius: 5px; cursor: pointer;">Try Again</button>
            </div>
        `;
    }
}

// Initialize when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
    window.captchaHandler = new SimpleCaptchaHandler();
});

// Additional protection for automation detection
(() => {
    'use strict';
    
    // Detect automation tools
    if (navigator.webdriver || window.phantom || window._phantom || window.callPhantom) {
        document.body.innerHTML = '<h1 style="text-align:center;padding:50px;">Automation detected. Access denied.</h1>';
        return;
    }
})();